<?LassoScript

	// This page allows a single poll to be viewed and edited.
	//
	// The page requires a parameter 'poll' which specifies the ID of the 
	// poll to be edited. A parameter 'action' determines what action to
	// perform on the poll.  Actions include:
	//
	// Update - The poll record and associated answer records are updated
	// according to the inputs provided by the user.  The results of the poll
	// are not changed.
	//
	// Archive - The poll is made inactive, the count for each answer record is
	// updated by counting the number of results, and all results are deleted.
	// Archived polls are no longer open for voting.
	//
	// Reset - The poll is made active, the count for each answer record is
	// set to zero, and all results are deleted.  This allows an archived poll
	// to be used again, but all results are lost.
	//
	// Delete - The poll, associated answer records, and all results are deleted.
	//
	// Cancel - No change is made to the poll and user is returned to the list.
	
	// Store the action and poll id in variables
	var: 'action' = (action_param: 'action');
	var: 'poll_id' = (action_param: 'poll');

	// Check the action
	if: ($action >> 'update');
	
		// On an update four tasks are performed.  First, the preferences are stored
		// in map.  Second, the answers for the poll are updated, created, or deleted.
		// Third, the current status of all the polls is updated.  Finally, the poll 
		// itself is updated.
		
		// Store the preferences in the preference map
		var: 'poll_prefs' = (map);		
		iterate: action_params, (var: 'temp');
			var: 'key' = $temp->first;
			var: 'value' = $temp->second;
			if: $key->(beginswith: 'prefs_');
				$key -= 'prefs_';
				$poll_prefs->(insert: $key = $value);
			/if;
		/iterate;
		
		// The action parameter 'answer' includes the ID for each answer which needs
		// to be updated.  If the ID contains 'new' then a new answer will be created.
		// Empty answers will be deleted (including all of their results).
		
		loop: (action_param: 'answers', -count);
			
			// Store the answer ID, answer text, and answer sort value in variables
			var: 'answer_id' = (action_param: 'answers', loop_count);
			var: 'answer_answer' = (action_param: 'answers_' + $answer_id + '_answer');
			var: 'answer_sort' = (action_param: 'answers_' + $answer_id + '_sort');
			
			// If the answer ID contains 'new' then create a new answer record.
			
			if: ($answer_id >> 'new');
				if: ($answer_answer != '');
					inline: -log=$log_level,
							-add,
							-database='op_quick_poll', 
							-table='answers',
							-keyfield='id',
							'poll_id'=$poll_id,
							'answer'=$answer_answer,
							'sort'=$answer_sort;
					/inline;
				/if;
			else;
				
				// If the answer text is not empty then update the answer record.
				// Otherwise the answer record is deleted.
				
				if: ($answer_answer != '');
					inline: -log=$log_level,
							-update,
							-database='op_quick_poll', 
							-table='answers',
							-keyfield='id',
							-keyvalue=$answer_id,
							'poll_id'=$poll_id,
							'answer'=$answer_answer,
							'sort'=$answer_sort;
					/inline;
				else;
					inline: -log=$log_level,
							-delete,
							-database='op_quick_poll', 
							-table='answers',
							-keyfield='id',
							-keyvalue=$answer_id;
					/inline;
				/if;
			/if;
		/loop;

		// If the user has chosen to make this the current poll then all of the
		// other polls need to be updated so there is only one current poll.
		
		if: (action_param: 'poll_current') == 'true';
			var: 'poll_current' = 'true';
			inline: -log=$log_level,
					-update,
					-database='op_quick_poll', 
					-table='polls',
					-keyfield='current',
					-keyvalue='true',
					'current'='false',
					-maxrecords='0';
			/inline;
		else;
			var: 'poll_current' = 'false';
		/if;

		// Update the poll record.  
		// Most of the values come from action parameters.
		// The poll preferences are stored as a serizlied map.
		
		inline: -log=$log_level,
				-update,
				-database='op_quick_poll', 
				-table='polls',
				-keyfield='id',
				-keyvalue=$poll_id,
				'title'=(action_param: 'poll_title'),
				'code'=(action_param: 'poll_code'),
				'status'=(action_param: 'poll_status'),
				'current'=(var: 'poll_current'),
				'question'=(action_param: 'poll_question'),
				'results'=(action_param: 'poll_results'),
				'display'=(action_param: 'poll_display'),
				'frequency'=(action_param: 'poll_frequency'),
				'prefs'=$poll_prefs->serialize->trim &;
		/inline;
		var: 'mode' = '';
	
	else: ($action >> 'reset');

		// On a reset each of the answer records is updated with a zero count,
		// all of the results for the poll are deleted, and the poll is made active.
		
		// Find the poll record
		inline: -log=$log_level,
				-search,
				-database='op_quick_poll',
				-table='polls',
				-keyfield='id',
				-keyvalue=$poll_id,
				-maxrecords=1,
				-returnfield='id';
			if: (found_count == 1);

				// Update all fo the answers of the poll with a vote count of zero
				inline: -log=$log_level,
						-update,
						-database='op_quick_poll',
						-table='answers',
						-keyfield='poll_id',
						-keyvalue=(field: 'id'),
						'result'=0,
						-maxrecords=0;
				/inline;

				// Delete all of the results records for the poll
				inline: -log=$log_level,
						-delete,
						-database='op_quick_poll',
						-table='results',
						-keyfield='poll_id',
						-keyvalue=(field: 'id');
				/inline;

				// Update the poll to make it active.
				inline: -log=$log_level,
						-update,
						-database='op_quick_poll',
						-table='polls',
						-keyfield='id',
						-keyvalue=(field: 'id'),
						'archive'='false';
				/inline;
			/if;
		/inline;
	
	else: ($action >> 'archive');

		// On an archive each of the answer records is updated with its vote count,
		// all of the results for the poll are deleted, and the poll is made inactive.
		
		// Find the poll record
		inline: -log=$log_level,
				-search,
				-database='op_quick_poll',
				-table='polls',
				-keyfield='id',
				-keyvalue=$poll_id,
				-maxrecords=1,
				-returnfield='id';
			if: (found_count == 1);
			
				// Find all of the answers for the poll
				inline: -log=$log_level,
						-search,
						-database='op_quick_poll',
						-table='answers',
						-keyfield='id',
						-op='eq', 'poll_id' = (field: 'id'),
						-maxrecords='all',
						-returnfield='id', -returnfield='result';
					records;
						
						// Find the total for each answer by counting the number of
						// results records.
						var: 'total' = 0;
						inline: -log=$log_level,
								-search,
								-database='op_quick_poll',
								-table='results',
								-keyfield='id',
								-op='eq', 'answer_id'=(field: 'id'),
								-maxrecords=1,
								-returnfield='id';
							var: 'total' = (found_count);
						/inline;
						
						// Update each answer record
						inline: -log=$log_level,
								-update,
								-database='op_quick_poll',
								-table='answers',
								-keyfield='poll_id',
								-keyvalue=(field: 'id'),
								'results'=$total;
						/inline;
					/records;
				/inline;
				
				// Delete all of the results records for the poll
				inline: -log=$log_level,
						-delete,
						-database='op_quick_poll',
						-table='results',
						-keyfield='poll_id',
						-keyvalue=(field: 'id');
				/inline;
				
				// Update the poll to make it inactive.
				inline: -log=$log_level,
						-update,
						-database='op_quick_poll',
						-table='polls',
						-keyfield='id',
						-keyvalue=(field: 'id'),
						'archive'='true';
				/inline;
			/if;
		/inline;
	
	else: ($action >> 'delete');

		// On a delete all of the answers and results for the poll are deleted and
		// the poll itself is deleted.
		
		// Find the poll record
		inline: -log=$log_level,
				-search,
				-database='op_quick_poll',
				-table='polls',
				-keyfield='id',
				-keyvalue=$poll_id,
				-maxrecords=1,
				-returnfield='id';
			if: (found_count == 1);
			
				// Delete all of the answers for the poll
				inline: -log=$log_level,
						-delete,
						-database='op_quick_poll',
						-table='answers',
						-keyfield='poll_id',
						-keyvalue=(field: 'id');
				/inline;

				// Delete all of the results for the poll
				inline: -log=$log_level,
						-delete,
						-database='op_quick_poll',
						-table='results',
						-keyfield='poll_id',
						-keyvalue=(field: 'id');
				/inline;

				// Delete the poll
				inline: -log=$log_level,
						-delete,
						-database='op_quick_poll',
						-table='polls',
						-keyfield='id',
						-keyvalue=(field: 'id');
				/inline;
			/if;
		/inline;
		var: 'mode' = '';
		var: 'poll_id' = '';
	
	else: ($action >> 'cancel');
	
		// On a cancel poll_id is set to empty which results
		// in the user being sent back to the list.
	
		var: 'poll_id' = '';
	
	/if;	
	
	// Search for the poll which needs to be displayed.  All of the fields
	// of the poll are transferred into variables which are then used for
	// all the inputs and selects in the update form.  The preferences are
	// deserialized back into a map.
	
	if: $poll_id != '';
		
		inline: -log=$log_level,
				-search,
				-database='op_quick_poll',
				-table='polls',
				-keyfield='id',
				-keyvalue=$poll_id,
				-maxrecords=1;
			if: (found_count > 0);
				var: 'handled' = true;
				var: 'poll_id' = (field: 'id');
				var: 'poll_title' = (field: 'title');
				var: 'poll_code' = (field: 'code');
				var: 'poll_question' = (field: 'question');
				var: 'poll_results' = (field: 'results');
				var: 'poll_display' = (field: 'display');
				var: 'poll_status' = (field: 'status');
				var: 'poll_archive' = (field: 'archive');
				var: 'poll_current' = (field: 'current');
				var: 'poll_frequency' = (field: 'frequency');
				if: (field: 'prefs') >> '<map';
					var: 'poll_prefs' = null;
					$poll_prefs->(deserialize: (field: 'prefs'));
				/if;
			/if;
		/inline;
		
	/if;
		
	// Define a helper tag [colorPicker].  This tag outputs a checkbox with a clickable 
	// label which allows a new color to be chosen.  By defining this custom tag the code
	// for each of the checkboxes can be modified in one place.
	
	var: 'colorPicker_id' = 0;
	define_tag: 'colorPicker', -required='name', -required='label';
		$colorPicker_id++;
		local: 'output' = '<input type="checkbox" name="prefs_' + #name + '" id="colorPicker_' + $colorPicker_id + '" value="' + $poll_prefs->(find: #name) + '"' + (($poll_prefs->keys >> #name)?' checked="checked"') + ' /> <a href="" onclick="return stylePicker(this,\'colorPicker_' + $colorPicker_id + '\',\'Type a color name or hex value:\',\'color\');"' + (($poll_prefs->keys >> #name)?' style="color: ' + $poll_prefs->(find: #name) + ';"') + '>' + #label + '</a>';
		return: @#output;
	/define_tag;

?>
[if: $handled == true]
	<h2>Lasso Quick Poll</h2>
	<div class="group_list" id="page_frame"[if: (var: 'ljax') == true] xmlns="http://www.w3.org/1999/xhtml"[/if]>
		<div class="group"><b class="rboth1 rrightclr"></b><b class="rboth2 rrightclr"></b><b class="rboth3 rrightclr"></b><b class="rboth4 rrightclr"></b>
			<div class="group_title">
				<a href="[response_filepath]">Home</a> >
				[var: 'poll_title']
			</div>
			<div class="item">
				<table width="100%">
					<tr>
						<td><script src="display/poll.lasso?code=[var: 'poll_code']" type="text/javascript" /></td>
						<td>[if: $poll_archive != 'true']<script src="display/poll.lasso?code=[var: 'poll_code']&amp;action=view" type="text/javascript" />[/if]</td>
					</tr>
				</table>
			</div>
			<form action="index.lasso" method="post">
				<div class="item">
					<input type="hidden" name="poll" value="[var: 'poll_id']" />
					<input type="hidden" name="mode" value="edit" />
					<table>
						<tr>
							<th>Summary:</th>
							<td>
								[if: $poll_status == 'true']This poll is visible.[else]This poll is currently hidden.  Only administrators will be able to see this poll.[/if]
								[if: $poll_current == 'true']This is the current poll.  It will show up by default if no code is specified
								in the &lt;script /&gt; block.[/if]
								[inline: -log=$log_level,
										-search,
										-database='op_quick_poll', 
										-table='answers',
										-keyfield='id',
										-op='eq', 'poll_id' = (var: 'poll_id'),
										-sortfield='sort',
										-maxrecords='all']
									[var: 'total' = 0][records][$total += (integer: (field: 'result'))][/records]
									[if: $total == 0]This poll does not have any votes yet.[else]This poll has [var: 'total'] votes.[/if]
								[/inline]
								[if: $poll_archive == true]
									This poll has been archived. It is no longer open for voting.
								[/if]
							</td>
						</tr>
						<tr>
							<th>Title:</th>
							<td>
								<input type="text" name="poll_title" value="[var: 'poll_title']" />
								[colorPicker: 'display_title_color','Text Color']
								[colorPicker: 'display_title_background','Background']
							</td>
						</tr>
						<tr>
							<th>Code:</th>
							<td>
								<input type="text" name="poll_code" value="[var: 'poll_code']" />
								<input type="checkbox" name="poll_current" value="true"[if: $poll_current == 'true'] checked="checked"[/if] />
								Current
							</td>
						</tr>
						<tr>
							<th>Visibility:</th>
							<td>
								<select name="poll_status">
									<option value="true"[if: $poll_status] selected="selected"[/if]>Show</option>
									<option value="false"[if: !$poll_status] selected="selected"[/if]>Hide</option>
								</select>
							</td>
						</tr>
						<tr>
							<th>Question:</th>
							<td><textarea name="poll_question" cols="40" rows="2">[var: 'poll_question']</textarea></td>
						</tr>
						<tr>
							<th>Style:</th>
							<td>
								[colorPicker: 'display_color','Default Text Color']
								[colorPicker: 'display_background','Poll Background']
							</td>
						</tr>
				[inline: -log=$log_level,
						-search,
						-database='op_quick_poll', 
						-table='answers',
						-keyfield='id',
						-op='eq', 'poll_id' = $poll_id,
						-maxrecords='all', 
						-sortfield='sort']
					[records]
						[var: 'loop_count' = loop_count]
						<tr>
							<th>[if: (loop_count == 1)]Answers:[/if]</th>
							<td>
								<input type="hidden" name="answers" value="[field: 'id']" />
								<input type="text" name="answers_[field: 'id']_answer" value="[field: 'answer']" />
								<select name="answers_[field: 'id']_sort">[loop: found_count]
									<option value="[loop_count]"[if: $loop_count == loop_count] selected="selected"[/if]>[loop_count]</option>[/loop]
								</select>
								[colorPicker: 'answers_' + (field: 'id') +'_color','Text Color']
							</td>
						</tr>
					[/records]
					[loop: (math_range: -min=0, -max=10, 10 - found_count)]
						[var: 'loop_count' = loop_count]
						<tr[if: (loop_count > 1)] style="display: none" id="new[loop_count]"[/if]>
							<th>[if: (loop_count == 1)]New Answers:[/if]</th>
							<td>
								<input type="hidden" name="answers" value="new[loop_count]" />
								<input type="text" name="answers_new[loop_count]_answer" value="" onchange="var temp = document.getElementById('new[loop_count + 1]'); if (temp && temp.style && temp.style.display) temp.style.display = 'visible';" />
								<input type="hidden" name="answers_new[loop_count]_sort" value="[loop_count + found_count]" />
							</td>
						</tr>
					[/loop]
				[/inline]
						<tr>
							<th>Display Message:</th>
							<td>
								<textarea name="poll_display" cols="40" rows="2">[var: 'poll_display']</textarea>
							</td>
						</tr>
						<tr>
							<th>Results Message:</th>
							<td>
								<textarea name="poll_results" cols="40" rows="2">[var: 'poll_results']</textarea>
							</td>
						</tr>
						<tr>
							<th>Frequency:</th>
							<td>Allow each user to vote: <select name="poll_frequency">
								<option value="once"[if: (var: 'poll_frequency') == 'once'] selected="selected"[/if]>Once</option>
								<option value="hour"[if: (var: 'poll_frequency') == 'hour'] selected="selected"[/if]>Hourly</option>
								<option value="day"[if: (var: 'poll_frequency') == 'day'] selected="selected"[/if]>Daily</option>
								<option value="month"[if: (var: 'poll_frequency') == 'month'] selected="selected"[/if]>Monthly</option>
							</select></td>
						</tr>
					</table>
					<p>
						<span style="float: right">
						[if: $poll_archive != 'true']
							<input type="submit" name="action" value="Archive Poll" onclick="return confirm('Once a poll has been archived its results can be viewed, but no more voting will be possible.');" />
							<input type="submit" name="action" value="Reset Results" onclick="return confirm('This will delete all of the poll\'s results.');" />
						[else]
							<input type="submit" name="action" value="Reset Poll" onclick="return confirm('This will reset the polls results and make it active again.');" />
						[/if]	
							<input type="submit" name="action" value="Delete Poll" onclick="return confirm('This will delete the poll and all of its results permanently.');" />
						</span>
						<input type="submit" name="action" value="Update Poll" />
						<input type="submit" name="action" value="Cancel" />
					</p>
					<br />
				</div>
			</form>
		</div>
		<div class="group">
			<div class="group_title"><a href="index.lasso">List Polls</a></div>
		</div>
	</div>
[/if]